//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.1
import QtGraphicalEffects 1.0

import wesual.data.Core 1.0
import wesual.Controls 1.0

Rectangle {
    id : priceRegion

    property list<Country> countryListing
    property int   currentIndex  : 0

    readonly property bool open : popup.popupVisible

    function openPopup()  {
        popup.activeIndex = -1;
        popup.show();
        popup.popupItem.focus = true;
    }

    activeFocusOnTab : true

    visible : countryListing.length > 1

    onOpenChanged : {
        if (!open && !popup.closedOnTab)
            priceRegion.forceActiveFocus();
        else
            popup.closedOnTab = false;
    }

    // Key handling
    Keys.onSpacePressed : {
        if (!popup.popupVisible) {
            priceRegion.openPopup();
        }
    }
    Keys.onDownPressed : {
        if (!open) {
            openPopup();
        }
    }

    implicitHeight : visible ? 26 : 0
    implicitWidth  : visible ? text.implicitWidth + indicator.width + 13 : 0
    z : popup.popupVisible ? 10000 : 0

    color : UI.color(UI.PrimaryBase)

    states : [
        State {
            name : "selected"
            when : ma.pressed || popup.popupVisible

            PropertyChanges {
                target : priceRegion
                color : UI.color(UI.PrimaryPress)
            }
        },
        State {
            name : "hovered"
            when : ma.containsMouse || activeFocus

            PropertyChanges {
                target : priceRegion
                color  : UI.color(UI.PrimaryHover)
            }
        }
    ]

    Text {
        id : text
        font {
            family : UI.fontFamily(UI.PrimaryFont)
            weight : UI.fontWeight(UI.PrimaryFont)
            pixelSize : 14
        }
        color : priceRegion.state === "selected" ?
                UI.color(UI.SelectedTextColor) :
                UI.color(UI.PrimaryTextColor)

        anchors {
            verticalCenter : parent.verticalCenter
            left : parent.left
            leftMargin : 5
        }
        text : {
            var currentCountry = countryListing[currentIndex];
            if (currentCountry) {
                var locale = Qt.locale(currentCountry.context.language);

                var symbol =
                        UI.currencyString(currentCountry.code,
                                     UI.fontFamily(UI.PrimaryFont));

                return currentCountry.name + " (" + symbol + ")";
            }
            return "";
        }
    }

    Rectangle {
        id : indicator

        anchors.right : parent.right
        height : parent.height
        width  : 15
        color : parent.color

        Image {
            id : image
            anchors.verticalCenter : parent.verticalCenter
            source : "qrc:/controls/icons/dropdown-indicator.png"
            width  : 10
            height :  5
            anchors {
                right : parent.right
                rightMargin : 5
                verticalCenter : parent.verticalCenter
            }
            visible : false
        }
        ColorOverlay {
            anchors.fill : image
            source : image
            color : ma.pressed || popup.popupVisible ?
                    UI.color(UI.PrimaryBase) : UI.color(UI.DropdownIndicatorIcon)
        }
    }

    MouseArea {
        id : ma
        anchors.fill : parent
        hoverEnabled : true
        onClicked : openPopup()
        cursorShape : Qt.PointingHandCursor
    }

    PriceRegionPopup {
        id : popup

        listing     : countryListing

        onAccepted : {
            priceRegion.currentIndex = result;
            popup.hide();
        }
    }
}
